<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main saor Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */

final class saor_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */

	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';


	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */

	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated

		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version

		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version

		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}


		// Add Plugin actions

		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );


        // Register widget scripts

		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);


		// Specific Register widget scripts

		// add_action( 'elementor/frontend/after_register_scripts', [ $this, 'saor_regsiter_widget_scripts' ] );
		// add_action( 'elementor/frontend/before_register_scripts', [ $this, 'saor_regsiter_widget_scripts' ] );


        // category register

		add_action( 'elementor/elements/categories_registered',[ $this, 'saor_elementor_widget_categories' ] );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'saor' ),
			'<strong>' . esc_html__( 'saor Core', 'saor' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'saor' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */

			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'saor' ),
			'<strong>' . esc_html__( 'saor Core', 'saor' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'saor' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(

			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'saor' ),
			'<strong>' . esc_html__( 'saor Core', 'saor' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'saor' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */

	public function init_widgets() {

		$widget_register = \Elementor\Plugin::instance()->widgets_manager;

		// Header Include file & Widget Register
		require_once( SAOR_ADDONS . '/header/header.php' );
		require_once( SAOR_ADDONS . '/header/saor-megamenu.php' );

		$widget_register->register ( new \saor_Header() );
		$widget_register->register ( new \saor_Megamenu() );


		// Include All Widget Files
		foreach($this->saor_Include_File() as $widget_file_name){
			require_once( SAOR_ADDONS . '/widgets/saor-'."$widget_file_name".'.php' );
		}
		// All Widget Register
		foreach($this->saor_Register_File() as $name){
			$widget_register->register ( $name );
		}
		
	}

	public function saor_Include_File(){
		return [
			'slider',
			'section-title',
			'progress',
			'image',
			'about',
			'button',
			'countdown',
			'sliding-text',
			'service',
			'mission',
			'team',
			'process',
			'animated-shape',
			'testimonial',
			'particale-shape',
			'client',
			'blog',
			'project',
			'price',
			'arrows',
			'contact',
			'video',
			'faq',
			'choose',
			'cta',
			'subscribe',
			'footer-about',
			'menu',
			'footer-contact',
			'footer-copyright',
			'contact-info',
			'team-info'
		];
	}

	public function saor_Register_File(){
		return [
			new \Saor_Banner(),
			new \Saor_Section_Title(),
			new \Saor_Progres_Bar(),
			new \Saor_Image(),
			new \Saor_About(),
			new \Saor_Button(),
			new \Saor_Countdown(),
			new \Saor_Sliding_Text(),
			new \Saor_Service(),
			new \Saor_Mission(),
			new \Saor_Team(),
			new \Saor_Process(),
			new \Saor_Animated_Shape(),
			new \Saor_Testimonial(),
			new \Saor_Particle_Shape(),
			new \Saor_Client_Logos(),
			new \Saor_Blog(),
			new \Saor_Project(),
			new \Saor_Price(),
			new \Saor_Arrows(),
			new \Saor_Contact_Form(),
			new \Saor_Video(),
			new \Saor_Faq(),
			new \Saor_Choose(),
			new \Saor_Cta(),
			new \Saor_Subscribe(),
			new \Saor_Footer_About(),
			new \Saor_Menu(),
			new \Saor_Footer_Contact,
			new \Saor_Footer_Copyright,
			new \Saor_Contact_Info,
			new \Saor_Team_Info,

		];
	}

    public function widget_scripts() {

  //       wp_enqueue_script(
  //           'saor-frontend-script',
  //           SAOR_PLUGDIRURI . 'assets/js/soar-frontend.js',
  //           array('jquery'),
  //           false,
  //           true
		// );

	}


    function saor_elementor_widget_categories( $elements_manager ) {

        $elements_manager->add_category(
            'saor',
            [
                'title' => __( 'Saor', 'saor' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
   
        $elements_manager->add_category(
            'saor_footer_elements',
            [
                'title' => __( 'Saor Footer Elements', 'saor' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'saor_header_elements',
            [
                'title' => __( 'Saor Header Elements', 'saor' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
	}
}

saor_Extension::instance();